import {type Router} from '../../types';
import {Layout} from '../../components/Layout';
import {Navigation} from '../../components/Navigation';
import {Main} from '../../components/Main';
import {css} from '../../css';
import {type Post} from './types';
import {CommentList} from './CommentList';
import {CommentForm} from './CommentForm';

const posts: Post[] = [
  {
    id: '1',
    title: 'Why TriFrost?',
    content: () => (
      <>
        <p>
          In "Why TriFrost?" we explore the motivation behind creating a framework that’s not tied to a single runtime or environment. We
          wanted something that felt familiar to Node.js developers, yet unlocked the performance and scaling benefits of emerging
          platforms.
        </p>
        <p>
          This post dives into the architectural decisions, our focus on middleware, and how TriFrost's minimal abstractions make it easy to
          plug into any modern toolchain or deployment setup.
        </p>
      </>
    ),
    comments: [],
  },
  {
    id: '2',
    title: 'Getting to v1.0',
    content: () => (
      <>
        <p>
          "Getting to v1.0" offers a behind-the-scenes look at our development roadmap. We discuss the features we prioritized, the hard
          lessons we learned in pre-1.0 releases, and the community feedback that has shaped our direction.
        </p>
        <p>
          Expect insights on improving developer experience, refining our TypeScript APIs, and why we believe v1.0 will set a new standard
          for backend frameworks across runtimes.
        </p>
      </>
    ),
    comments: [],
  },
  {
    id: '3',
    title: 'Observability as a First-Class Citizen',
    content: () => (
      <>
        <p>
          "Observability as a First-Class Citizen" explains why we integrated OpenTelemetry into TriFrost from the very start. We believe
          that modern backends deserve production-grade tracing and metrics, not as an afterthought but as a built-in feature.
        </p>
        <p>
          This post unpacks how TriFrost’s tracing hooks work, how to export spans to SigNoz or other tools, and how to use this data to
          understand and improve your app’s performance.
        </p>
      </>
    ),
    comments: [],
  },
];

export async function blogRouter(r: Router) {
  r.get('', async ctx => {
    return ctx.html(
      <Layout title="TriFrost Blog">
        <>
          <Navigation active="blog" />
          <Main>
            <h1>TriFrost Blog</h1>
            <p className={css({marginTop: css.$v.space_l})}>
              Welcome to the TriFrost Blog — your inside look at the evolution of our framework, the ideas driving our roadmap, and the
              technical deep-dives that power our ecosystem.
            </p>
            <p className={css({marginTop: css.$v.space_l})}>
              Here we share everything from release announcements and architectural patterns to performance tips and real-world use cases.
              Stay tuned to learn how TriFrost is reshaping backend development across modern runtimes.
            </p>
            <div className={css.use('f', 'fv', {marginTop: css.$v.space_xl, gap: css.$v.space_m})}>
              {posts.map(el => (
                <article>
                  <a
                    href={`/blog/${el.id}`}
                    className={css.use('f', 'panel', {
                      padding: css.$v.space_l,
                      color: css.$t.fg,
                      textDecoration: 'none',
                      [css.hover]: {textDecoration: 'underline'},
                    })}
                  >
                    <h2>{el.title}</h2>
                  </a>
                </article>
              ))}
            </div>
          </Main>
        </>
      </Layout>,
    );
  })
    .get('/:postId', async ctx => {
      const post = posts.find(p => p.id === ctx.state.postId);
      if (!post) return ctx.text('Post not found', {status: 404});

      return ctx.html(
        <Layout title={`TriFrost: ${post.title}`}>
          <>
            <Navigation active="blog" />
            <Main>
              <h1>{post.title}</h1>
              <div className={css.use('f', 'fv', {gap: css.$v.space_l, marginTop: css.$v.space_l})}>{post.content()}</div>
              <h2 className={css({marginTop: css.$v.space_xl})}>Comments</h2>
              <CommentForm post={post} />
              <CommentList post={post} inject={true} />
            </Main>
          </>
        </Layout>,
      );
    })
    .post('/:postId/comment', async ctx => {
      const {comment} = ctx.body as {comment: string};
      const post = posts.find(p => p.id === ctx.state.postId);
      if (!post) return ctx.status(404);

      post.comments.unshift({id: String(post.comments.length + 1), body: comment, on: new Date()});
      return ctx.html(<CommentList post={post} inject={false} />);
    })
    .del('/:postId/comment/:commentId', async ctx => {
      const post = posts.find(p => p.id === ctx.state.postId);
      if (!post) return ctx.status(404);

      post.comments = (post?.comments || []).filter(el => el.id !== ctx.state.commentId);
      return ctx.html(<CommentList post={post} inject={false} />);
    });
}
