import {App, DurableObjectCache, JsonExporter} from '@trifrost/core';
import {Layout} from './components/Layout';
import {TodoForm, TodoList, type Todo} from './components/Todos';
import {type Env} from './types';
import {css} from './css';

export {TriFrostDurableObject} from '@trifrost/core';

const app = await new App<Env>({
  client: {css},
  cache: ({env}) => new DurableObjectCache({store: env.MAIN_DURABLE}),
  tracing: {
    exporters: ({env}) => [
      new JsonExporter(),
      /* Todo: Once you have your uptrace dsn filled in in .dev.vars, enable here */
      // new OtelHttpExporter({
      // 	logEndpoint: 'https://otlp.uptrace.dev/v1/logs',
      // 	spanEndpoint: 'https://otlp.uptrace.dev/v1/traces',
      // 	headers: {'uptrace-dsn': env.UPTRACE_DSN},
      // }),
    ],
  },
})
  .get('/', async ctx => {
    const todos = (await ctx.cache.get<Todo[]>('todos')) || [];

    return ctx.html(
      <Layout>
        <main
          className={css({
            width: '100%',
            maxWidth: '50rem',
            flexGrow: '1',
          })}
        >
          <h1 style={css.mix('title')}>📝 TriFrost: HTMX Todos</h1>
          <TodoForm />
          <TodoList todos={todos} />
        </main>
      </Layout>,
    );
  })
  .post('/', async ctx => {
    let todos = (await ctx.cache.get<Todo[]>('todos')) || [];

    const text = String((ctx.body as {text: string}).text || '');
    if (todos.findIndex(el => el.text === text) < 0) {
      todos.push({id: crypto.randomUUID(), text});
      todos = todos.slice(0, 50);
      await ctx.cache.set('todos', todos);
    }
    return ctx.html(<TodoList todos={todos} />);
  })
  .del('/:id', async ctx => {
    let todos = (await ctx.cache.get<Todo[]>('todos')) || [];

    todos = todos.filter(t => t.id !== ctx.state.id);

    await ctx.cache.set('todos', todos);
    return ctx.html(<TodoList todos={todos} />);
  })
  .post('/complete', async ctx => {
    const todos = (await ctx.cache.get<Todo[]>('todos')) || [];

    const {done} = ctx.body as {done?: string | string[]};
    const to_complete = new Set(typeof done === 'string' ? [done] : Array.isArray(done) ? done : []);
    if (to_complete.size) {
      for (const todo of todos) {
        if (to_complete.has(todo.id)) todo.completed = true;
      }
      await ctx.cache.set('todos', todos);
    }
    return ctx.html(<TodoList todos={todos} />);
  })
  .boot();

export default app;
